/*
   This program runs a child process with the child's
   standard input and output streams redirected to files.

   Important: This only works with Java 7. The class and
              methods used here were only recently added to Java!

   Important: Notice that this program is logically the
   same as the following command line.

   C:\> java Double < L50.txt > test.txt

   In the case of the above command line, the shell
   program (cmd.exe) sets up the I/O redirection before
   running the program Double.class. In the case of this
   program, this program itself runs Double.class and this
   program sets up the I/O redirection for its child. So
   this program is acting like a very simple shell program.
*/
import java.io.*;

public class Java7_RedirectChildStdinStdoutToFiles
{
   public static void main(String[] args) throws IOException, InterruptedException
   {
      // Create a command line for running the child.
      ProcessBuilder pb = new ProcessBuilder("java", "Double");

      // Before starting the child, tell ProcessBuilder to redirect its stdin to a file
      pb.redirectInput( ProcessBuilder.Redirect.from( new File( "L50.txt" ) ) );
      // Before starting the child, tell ProcessBuilder to redirect its stdout to a file
      pb.redirectOutput( ProcessBuilder.Redirect.to( new File( "test.txt" ) ) );
      // Tell ProcessBuilder to redirect the stderr stream to be the same as stdout.
      pb.redirectErrorStream(true);

      // Now run the child process.
      Process process = pb.start();  // this throws IOException
      // Wait for the child to do all of its work.
      process.waitFor();  // this throws InterruptedException

      // We are done.
   }
}

/*
   IMPORTANT:
   Comment out the line that redirects stdout to a file.
   When you run the program, it will hang and never terminate.
   Change the input file to L25.txt, recompile, and run the
   program again. Now it will terminate. What is going on here?

   If you do not redirect the child's stdout, the data written
   to stdout just goes to some buffer somewhere. The buffer is
   not very big. The file L50.txt more than fills up this buffer.
   When the buffer is full, the child process's writes to stdout
   block! The operating system (or maybe the Java runtime) is
   waiting for someone to remove something from the buffer before
   it lets the child process write more data to the buffer. But no
   other process is removing anything from the buffer. So the child
   just hangs. If you make the input file L25.txt, it is smaller
   than the output buffer, so the program runs to completion.
*/